{ This program converts a plain text file describing a model into a file
  called equations.pas. The equations.pas file is then used in modelshell to
  create a new model.
  First a blank equations.pas file is read into a memo component (MmEquations). 
  The user must specify the text or rich text file describing the new model. This
  text file is then read into the richedit component (ReModelDef). The user
  then clicks on the Create code button (btnCreatecode) and the information in
  the rich edit is copied to the appropriate places in the equations memo. Once
  complete, the equations memo is saved to the same directory as the model
  description text file. }
unit CrEqmain;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, stypes, ComCtrls;

const
  maxArrays = 100;

type
  TPart = (stBeginning, stEnd);
  Tshortstring = string[stringlength];
  TArrayInfo = record
    VarName: Tshortstring;
    NumElements: integer;
  end;
  TArrayList = array[1..MaxArrays] of TArrayInfo;

type
  TFmMain = class(TForm)
    MmInstruct: TMemo;
    LblFilename: TLabel;
    EdFilename: TEdit;
    BtnCreatecode: TButton;
    BtnClose: TButton;
    DlgOpenModelDef: TOpenDialog;
    MmEquations: TMemo;
    ReModelDef: TRichEdit;
    procedure ChooseDefFile(Sender: TObject);
    procedure BtnCreatecodeClick(Sender: TObject);
    procedure BtnCloseClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
  private
    { Private declarations }
    FDefPath:string;
    FExePath:string;
    tempmodeldef:Tmodeldef;
    tempstate:statearray;
    tempprocess:processarray;
    tempdrive:drivearray;
    temppar:paramarray;
    numOther:integer;
    tempother:paramarray;
    ModelDefFilename:String;
    arrays: TArrayList;
    TotalNumArrays: integer;
    procedure DefModel;
    procedure DefStates;
    procedure DefProcess;
    procedure DefDrive;
    procedure DefOtherVar;
    procedure GetFunctions;
    procedure CreateProcessesProc;
    function LineSearch(tempmemo: TCustomMemo; Sstring:string):integer;
    function DivideString(fromString, divider:string; WhichPart:TPart):string;
    procedure ParseString(var tstring, tname, tunits, tsymbol: string;
                          var tnpar:integer; var tptype:processtype; var narray: integer);
    function ParCount(processnum:integer):integer;
    function GetArrayNum(varname:string): integer;
    function CheckDerivatives(fline: integer): integer;
    function RemoveTrailingComma(lnum: integer): integer;
    procedure RemoveFinalDouble;
    function RemoveArrayIndex(tstring: string): string;
  public
    { Public declarations }
  end;

var
  FmMain: TFmMain;

implementation

{$R *.DFM}

{ }
procedure TFmMain.ChooseDefFile(Sender: TObject);
var
 oldfilename : string;
begin
 oldfilename := ModelDefFilename;
 // If the user typed directly in the box
 if Sender is TEdit then ModelDefFilename := EdFilename.text  // Set filename
 else // The user clicked either the menu or the label
   begin  // Show the open file dialog
           // First set the dialog box default filename to the current file
      DlgOpenModelDef.filename := ModelDefFilename;
          // If the user chooses OK in the dialog then set the new paramfilename
      if DlgOpenModelDef.execute then ModelDefFilename := DlgOpenModelDef.filename;
   end;
 EdFilename.Text := ModelDefFilename;
 FDefPath := ExtractFilePath(ModelDefFilename);
 if ModelDefFilename <> '' then BtnCreateCode.Enabled := True;
end;

{ This procedure converts the description in the model description file into
  pascal code and places it in the appropriate place in the equations file. }
procedure TFmMain.BtnCreatecodeClick(Sender: TObject);
begin
 if ModelDefFilename <> '' then
  begin
    BtnCreateCode.Enabled := False;
    MmEquations.Clear;
    MmEquations.lines.LoadFromFile(FExePath + 'equationsblank.pas');
    ReModelDef.Clear;
    ReModelDef.Lines.LoadFromFile(ModelDefFilename);
    TotalNumArrays := 1;
    DefModel;
    DefStates;
    DefProcess;
    DefDrive;
    DefOtherVar;
    CreateProcessesProc;
    MmEquations.Lines.SavetoFile(FDefPath + 'equations.pas');
    BtnCreateCode.Enabled := True;
    MessageDlg('Your model is now ready to run. Using Windows Explorer, find '
       + 'and double click on the file modelshell6445.dpr, your model will open '
       + 'inside Delphi. Click on the green triangle in the menu bar to run '
       + 'your model.', mtInformation, [mbOK], 0)
  end
 else
  MessageDlg('Invalid Model Definition File. Please reenter',
              mtWarning, [mbOK], 0);
end;

{ Determines the model name, version, and time unit based on the file read into
  the ReModelDef memo component. The Model name, version and time unit are
  copied to the  MmEquations memo and to the tempModelDef record. }
procedure TFmMain.DefModel;
var
 tempstring:string;
 fromLineNum, toLineNum:integer;
begin
 fromLineNum := LineSearch(ReModelDef,'Model Name');
 tempModelDef.modelname :=DivideString(ReModelDef.Lines[fromLineNum],'=',stEnd);
 if length(tempModelDef.modelname) > stringlength then
    raise Exception.Create('Model name is too long. Decrease model name to 25 characters or less and rerun Create Equations.');

 tempstring := 'ModelDef.modelname := ''' + tempModelDef.modelname + ''';';
 toLineNum := LineSearch(MmEquations,'ModelDef.modelname :=');
 MmEquations.Lines[toLineNum] := tempstring;

 fromLineNum := LineSearch(ReModelDef,'Model Version');
 tempModelDef.versionnumber :=
                          DivideString(ReModelDef.Lines[fromLineNum],'=',stEnd);
 tempstring := 'ModelDef.versionnumber := ''' + tempModelDef.versionnumber
                  + ''';';
 toLineNum := LineSearch(MmEquations,'ModelDef.versionnumber :=');
 MmEquations.Lines[toLineNum] := tempstring;

 fromLineNum := LineSearch(ReModelDef,'Model Time Unit');
 tempModelDef.timeunit := DivideString(ReModelDef.Lines[fromLineNum],'=',stEnd);
 tempstring := 'ModelDef.timeunit := ''' + tempModelDef.timeunit + ''';';
 toLineNum := LineSearch(MmEquations,'ModelDef.timeunit :=');
 MmEquations.Lines[toLineNum] := tempstring;

 fromLineNum := LineSearch(ReModelDef,'Model Contact');
 tempModelDef.contactperson :=DivideString(ReModelDef.Lines[fromLineNum],'=',stEnd);
 tempstring := 'ModelDef.contactperson := ''' + tempModelDef.contactperson + ''';';
 toLineNum := LineSearch(MmEquations,'ModelDef.contactperson :=');
 MmEquations.Lines[toLineNum] := tempstring;

 fromLineNum := LineSearch(ReModelDef,'Model Contact Address Line 1');
 tempModelDef.contactaddress1 :=DivideString(ReModelDef.Lines[fromLineNum],'=',stEnd);
 tempstring := 'ModelDef.contactaddress1 := ''' + tempModelDef.contactaddress1 + ''';';
 toLineNum := LineSearch(MmEquations,'ModelDef.contactaddress1 :=');
 MmEquations.Lines[toLineNum] := tempstring;

 fromLineNum := LineSearch(ReModelDef,'Model Contact Address Line 2');
 tempModelDef.contactaddress2 :=DivideString(ReModelDef.Lines[fromLineNum],'=',stEnd);
 tempstring := 'ModelDef.contactaddress2 := ''' + tempModelDef.contactaddress2 + ''';';
 toLineNum := LineSearch(MmEquations,'ModelDef.contactaddress2 :=');
 MmEquations.Lines[toLineNum] := tempstring;

 fromLineNum := LineSearch(ReModelDef,'Model Contact Address Line 3');
 tempModelDef.contactaddress3 :=DivideString(ReModelDef.Lines[fromLineNum],'=',stEnd);
 tempstring := 'ModelDef.contactaddress3 := ''' + tempModelDef.contactaddress3 + ''';';
 toLineNum := LineSearch(MmEquations,'ModelDef.contactaddress3 :=');
 MmEquations.Lines[toLineNum] := tempstring;
end;

{ Defines the state variables of the model based on the file read into the
  ReModelDef memo.  }
procedure TFmMain.DefStates;
var
 i, j, m, num, fromline, toline, tnpar, tnarray:integer;
 tempstring, tname, tunits, tsymbol: string;
// tname, tunits, tsymbol: Tshortstring;
 tptype:processtype;
begin
// Define the state variables
 fromline := LineSearch(ReModelDef, 'States');
 // Advance 1 lines in the memo to get to the line above the state variable listing
 fromline := fromline + 2;
 tempstring := ReModelDef.lines[fromline];
 repeat
   tempstring := ReModelDef.lines[fromline];
   tempstring := trim(tempstring);
   fromline := fromline + 1;
 until tempstring <> '';
 i := 1;
 while tempstring <> 'End' do
   begin
     if i > maxstate then
       raise Exception.Create('Too many state variables. Increase maxstate in stypes.');
     ParseString(tempstring, tname, tunits, tsymbol, tnpar, tptype, tnarray);
     if tname[1] <> '*' then       // Not an array variable
       begin
         m := length(tname);
         delete(tname, stringlength-1, m-stringlength+2);
         tempstate[i].name := tname;
         tempstate[i].units := tunits;
         tempstate[i].symbol := tsymbol;
         i := i + 1;
       end
     else                         // An array variable
       begin
         for j := 1 to tnarray do
           begin
             m := length(tname);
             delete(tname, stringlength-1, m-stringlength+2);
             tempstate[i + j - 1].name := tname + inttostr(j);
             tempstate[i + j - 1].units := tunits;
             tempstate[i + j - 1].symbol := tsymbol + '[' + inttostr(j) + ']';
           end;
         if TotalNumArrays > MaxArrays then
           raise Exception.Create('Too many arrays. Increase MaxArrays.');
         arrays[TotalNumArrays].VarName := tname; // Used in createprocessesproc procedure
         arrays[TotalNumArrays].NumElements := tnarray;
         TotalNumArrays := TotalNumArrays + 1;
         i := i + tnarray;
       end;
     repeat
      tempstring := ReModelDef.lines[fromline];
      tempstring := trim(tempstring);
      fromline := fromline + 1;
     until tempstring <> '';
   end;
 tempModelDef.numstate := i - 1;     // Need tempModelDef for the equations

// Update the MmEquations memo
 toline := LineSearch(MmEquations,'ModelDef.numstate :=');
 tempstring := 'ModelDef.numstate := ' + inttostr(tempModelDef.numstate) + ';';
 MmEquations.lines[toline] := tempstring;

 toline := LineSearch(MmEquations,'with stat[1] do');
 for i := 1 to 4 do MmEquations.Lines.Delete(toline); // Delete the example code

 i := 1;
 j := 1;
 repeat
   MmEquations.Lines.Insert(toline, ' ');
   MmEquations.Lines.Insert(toline + 1,'with stat[' + inttostr(i) + '] do');
   MmEquations.Lines.Insert(toline + 2, ' begin');
   if tempstate[i].name[1] <> '*' then
     begin
       tempstring := '    name:=''' + tempstate[i].name + ''';  '
                     + 'units:=''' + tempstate[i].units + ''';  '
                     + 'symbol:=''' + tempstate[i].symbol + ''';';
     end
   else
     begin
       m := GetArrayNum(tempstate[i].name);
       num := arrays[m].NumElements;
       tempstring := '    name:=''' + tempstate[i].name + ''';  '
                     + 'units:=''' + tempstate[i].units + ''';  '
                     + 'symbol:=''' + tempstate[i].symbol + ''';';
       j := j + 1;
       if (j = num + 1) then j := 1;
     end;
   MmEquations.Lines.Insert(toline + 3, tempstring);
   MmEquations.Lines.Insert(toline + 4, ' end;');
   toline := toline + 5;
   i := i + 1;
 until i = tempModelDef.numstate + 1;
end;

{ Defines the processes and parameters of the model based on the file read into
  the ReModelDef memo. }
procedure TFmMain.DefProcess;
var
 i, j, k, m, num, fromline, toline, tnpar, numprevpar, tnarray, numelements, arraynum:integer;
 tempstring, tname, tunits, tsymbol: string;
 tptype:processtype;
 holdprocess: Tprocessvariable;
 holdparam: array[1..15] of Tparamvariable;
begin
 { The first numstate processes are the derivatives of the state variables.
   These processes aren't used in this procedure but are used in the
   CreateProcessesProc. }
 i := 1;
 repeat
  if tempstate[i].name[1] <> '*' then
   begin // Not an array variable
    tempprocess[i].name := 'd'+tempstate[i].name + 'dt';
    tempprocess[i].units := tempstate[i].units + 't-1';
    tempprocess[i].symbol := 'd' + tempstate[i].symbol + 'dt';
    i := i + 1;
   end
  else // An array variable
   begin
    arraynum := GetArrayNum(tempstate[i].name);
    tname := tempstate[i].name;
    numelements := arrays[arraynum].NumElements;
    if numelements < 10 then
      delete(tname, length(tname), 1)
    else
      delete(tname, length(tname)-1, 1);
    delete(tname, 1, 1);
    tsymbol := tempstate[i].symbol;
    if numelements < 10 then
      delete(tsymbol, length(tsymbol)-2, 3)
    else
      delete(tsymbol, length(tsymbol)-3, 4);
    for j := 1 to numelements do
     begin
      tempprocess[i+j-1].name := '*d' + tname + 'dt' + inttostr(j);
      tempprocess[i+j-1].units := tempstate[i].units + 't-1';
      tempprocess[i+j-1].symbol := 'd' + tsymbol + 'dt' +'[' + inttostr(j) + ']';
     end;
    i := i + arrays[arraynum].NumElements;
   end;
 until i = tempModelDef.numstate + 1;

 // Define the processes
 fromline := LineSearch(ReModelDef, 'Process');
 // Advance 2 lines in the memo to get from numprocess to the line above the variable listing.
 fromline := fromline + 3;
 tempstring := ReModelDef.lines[fromline];
 repeat
   tempstring := ReModelDef.lines[fromline];
   tempstring := trim(tempstring);
   fromline := fromline + 1;
 until tempstring <> '';
 i := tempmodeldef.numstate + 1;
 while tempstring <> 'End' do
 begin
  if i > maxprocess then
     raise Exception.Create('Too many processes. Increase maxprocess in stypes.');
  ParseString(tempstring, tname, tunits, tsymbol, tnpar, tptype, tnarray);
  if tname[1] <> '*' then        // Not an array variable
    begin
      m := length(tname);
      delete(tname, stringlength-1, m-stringlength+2);
      tempprocess[i].name := tname;
      tempprocess[i].units := tunits;
      tempprocess[i].symbol := tsymbol;
      tempprocess[i].parameters := tnpar;
      tempprocess[i].ptype := tptype;
      numprevpar := ParCount(i);
      for j := 1 to tnpar do
        begin
          if numprevpar + j > maxparam then
             raise Exception.Create('Too many parameters. Increase maxparam in stypes.');
          repeat
            tempstring := ReModelDef.lines[fromline];
            tempstring := trim(tempstring);
            fromline := fromline + 1;
          until tempstring <> '';
          ParseString(tempstring, tname, tunits, tsymbol, num, tptype, tnarray);
          temppar[numprevpar + j].name := tname;
          temppar[numprevpar + j].units := tunits;
          temppar[numprevpar + j].symbol := tsymbol;
        end;
      i := i + 1;
    end
  else                          // An array variable
    begin
     for k := 1 to tnarray do
       begin
        if k = 1 then
         begin
          m := length(tname);
          delete(tname, stringlength-1, m-stringlength+2);
          tempprocess[i + k - 1].name := tname;
          tempprocess[i + k - 1].units := tunits;
          tempprocess[i + k - 1].symbol := tsymbol;
          tempprocess[i + k - 1].parameters := tnpar;
          tempprocess[i + k - 1].ptype := tptype;
          holdprocess := tempprocess[i + k - 1];
          tempprocess[i + k - 1].name := tname + inttostr(k);
          tempprocess[i + k - 1].symbol := tsymbol+ '[' + inttostr(k) + ']';
          numprevpar := ParCount(i + k - 1);
          arrays[TotalNumArrays].VarName := tname; // Used in createprocessesproc procedure
          arrays[TotalNumArrays].NumElements := tnarray;
          TotalNumArrays := TotalNumArrays + 1;
          if tnpar > 15 then  // To change this value, change the variable declaration for holdparam
            raise exception.Create('Process, ' +
              tempprocess[i + k - 1].name + ', has too many parameters.');
          for j := 1 to tnpar do
            begin
              if numprevpar + j > maxparam then
                raise Exception.Create('Too many parameters. Increase maxparam in stypes.');
              repeat
                tempstring := ReModelDef.lines[fromline];
                tempstring := trim(tempstring);
                fromline := fromline + 1;
              until tempstring <> '';
              ParseString(tempstring, tname, tunits, tsymbol, num, tptype, tnarray);
              m := length(tname);
              delete(tname, stringlength-1, m-stringlength+2);
              temppar[numprevpar + j].name := tname;
              temppar[numprevpar + j].units := tunits;
              temppar[numprevpar + j].symbol := tsymbol;
              holdparam[j] := temppar[numprevpar + j];
              temppar[numprevpar + j].name := tname + inttostr(k);
              temppar[numprevpar + j].symbol := tsymbol + '[' + inttostr(k) + ']';
            end;
         end
        else // Not the first process of an array so use saved info instead of reading from file
         begin
          tempprocess[i + k - 1] := holdprocess;
          tempprocess[i + k - 1].name := holdprocess.name + inttostr(k);
          tempprocess[i + k - 1].symbol := holdprocess.symbol + '[' + inttostr(k) + ']';
          numprevpar := ParCount(i + k - 1);
          for j := 1 to holdprocess.parameters do
            begin
              if numprevpar + j > maxparam then
                raise Exception.Create('Too many parameters. Increase maxparam in stypes.');
              temppar[numprevpar + j] := holdparam[j];
              temppar[numprevpar + j].name := temppar[numprevpar + j].name
                     + inttostr(k);
              temppar[numprevpar + j].symbol := temppar[numprevpar + j].symbol
                     + '[' + inttostr(k) + ']';
            end;
         end;
       end;
     if TotalNumArrays > MaxArrays then
          raise Exception.Create('Too many arrays. Increase MaxArrays.');
    end;
  i := i + tnarray;
  repeat
    tempstring := ReModelDef.lines[fromline];
    tempstring := trim(tempstring);
    fromline := fromline + 1;
  until tempstring <> '';
 end;
 tempModelDef.numprocess := i - tempModelDef.numstate - 1;     // Need tempModelDef for the equations
 tempModelDef.numparam := ParCount(tempmodeldef.numstate +
                                                   tempmodeldef.numprocess);

// Update the MmEquations memo
 toline := LineSearch(MmEquations,'ModelDef.numprocess :=');
 tempstring := 'ModelDef.numprocess := ModelDef.numstate + ' +
                                       inttostr(tempModelDef.numprocess) + ';';
 MmEquations.lines[toline] := tempstring;

 toline := LineSearch(MmEquations,'CurrentProc := ModelDef.numstate');
 for i := 1 to 15 do MmEquations.Lines.Delete(toline); // Delete example code.
 i := 1;
 repeat
   MmEquations.Lines.Insert(toline, ' ');
   MmEquations.Lines.Insert(toline + 1, 'CurrentProc := ModelDef.numstate + '
                                                   + inttostr(i) + ';');
   MmEquations.Lines.Insert(toline + 2, 'With proc[CurrentProc] do');
   MmEquations.Lines.Insert(toline + 3, '   begin');
   MmEquations.Lines.Insert(toline + 4, '      name       := ''' +
               tempprocess[tempmodeldef.numstate + i].name + ''';');
   MmEquations.Lines.Insert(toline + 5, '      units       := ''' +
               tempprocess[tempmodeldef.numstate + i].units + ''';');
   MmEquations.Lines.Insert(toline + 6, '      symbol       := ''' +
               tempprocess[tempmodeldef.numstate + i].symbol + ''';');
   MmEquations.Lines.Insert(toline + 7, '      parameters       := ' +
           inttostr(tempprocess[tempmodeldef.numstate + i].parameters) + ';');
   case tempprocess[tempmodeldef.numstate + i].ptype of
     ptGroup1: tempstring := 'ptGroup1';
     ptGroup2: tempstring := 'ptGroup2';
     ptGroup3: tempstring := 'ptGroup3';
     ptGroup4: tempstring := 'ptGroup4';
     ptGroup5: tempstring := 'ptGroup5';
   else
     tempstring := 'ptGroup1';
   end;
   MmEquations.Lines.Insert(toline + 8, '      ptype       := ' + tempstring
     + ';');
   MmEquations.Lines.Insert(toline + 9, '   end;');
   numprevpar := ParCount(tempmodeldef.numstate + i);
   toline := toline + 10;
   if tempprocess[tempmodeldef.numstate + i].parameters > 0 then
     begin
      MmEquations.Lines.Insert(toline, 'npar:=ParCount(CurrentProc);');
      for m := 1 to tempprocess[tempmodeldef.numstate + i].parameters do
       begin
         MmEquations.Lines.Insert(toline + 1, 'with par[npar + ' +
                                                   inttostr(m) + '] do');
         MmEquations.Lines.Insert(toline + 2, ' begin');
         tempstring := '    name:=''' + temppar[numprevpar + m].name + ''';  '
                         + 'units:=''' + temppar[numprevpar + m].units + ''';  '
                         + 'symbol:=''' + temppar[numprevpar + m].symbol + ''';';
         MmEquations.Lines.Insert(toline + 3, tempstring);
         MmEquations.Lines.Insert(toline + 4, ' end;');
         toline := toline + 4;
       end;                       
      toline := toline + 1;
     end;
   i := i + 1;
 until i = tempModelDef.numprocess + 1;
end;

{ Defines the driver variables of the model based on the file read into
  the ReModelDef memo. }
procedure TFmMain.DefDrive;
var
 i, j, m, fromline, toline, tnpar, tnarray:integer;
 tempstring, tname, tunits, tsymbol: string;
// tname, tunits, tsymbol: Tshortstring;
 tptype:processtype;
begin
// Define the driver variables
 fromline := LineSearch(ReModelDef, 'Drive');
 // Advance 1 line in the memo to get to the line above the driver listing
 fromline := fromline + 2;
 tempstring := ReModelDef.lines[fromline];
 repeat
   tempstring := ReModelDef.lines[fromline];
   tempstring := trim(tempstring);
   fromline := fromline + 1;
 until tempstring <> '';
 i := 1;
 while tempstring <> 'End' do
  begin
   if i > maxdrive then
      raise Exception.Create('Too many drivers. Increase maxdrive in stypes.');
   ParseString(tempstring, tname, tunits, tsymbol, tnpar, tptype, tnarray);
   if tname[1] <> '*' then     // Not an array variable
     begin
       m := length(tname);
       delete(tname, stringlength-1, m-stringlength+2);
       tempdrive[i].name := tname;
       tempdrive[i].units := tunits;
       tempdrive[i].symbol := tsymbol;
       i := i + 1;
     end
   else                   // An array variable
     begin
       for j := 1 to tnarray do
         begin
          m := length(tname);
          delete(tname, stringlength-1, m-stringlength+2);
          tempdrive[i + j - 1].name := tname + inttostr(j);
          tempdrive[i + j - 1].units := tunits;
          tempdrive[i + j - 1].symbol := tsymbol + '[' + inttostr(j) + ']';
         end;
       arrays[TotalNumArrays].VarName := tname; // Used in createprocessesproc procedure
       arrays[TotalNumArrays].NumElements := tnarray;
       TotalNumArrays := TotalNumArrays + 1;
       if TotalNumArrays > MaxArrays then
           raise Exception.Create('Too many arrays. Increase MaxArrays.');
       i := i + tnarray;
     end;
   repeat
     tempstring := ReModelDef.lines[fromline];
     tempstring := trim(tempstring);
     fromline := fromline + 1;
   until tempstring <> '';
  end;
 tempModelDef.numdrive := i - 1;     // Need tempModelDef for the equations

// Update the MmEquations memo
 toline := LineSearch(MmEquations,'ModelDef.numdrive :=');
 tempstring := 'ModelDef.numdrive := ' + inttostr(tempModelDef.numdrive) + ';';
 MmEquations.lines[toline] := tempstring;

 toline := LineSearch(MmEquations,'with drive[1] do');
 for i := 1 to 4 do MmEquations.Lines.Delete(toline); // Delete the example code

 for i := 1 to tempModelDef.numdrive do
   begin
     MmEquations.Lines.Insert(toline, ' ');
     MmEquations.Lines.Insert(toline + 1, 'with drive[' + inttostr(i) + '] do');
     MmEquations.Lines.Insert(toline + 2, ' begin');
     tempstring := '    name:=''' + tempdrive[i].name + ''';  '
                       + 'units:=''' + tempdrive[i].units + ''';  '
                       + 'symbol:=''' + tempdrive[i].symbol + ''';';
     MmEquations.Lines.Insert(toline + 3, tempstring);
     MmEquations.Lines.Insert(toline + 4, ' end;');
     toline := toline + 5;
   end;
end;

{ Defines the other variables (non output variables) of the model based on the file read into
  the ReModelDef memo. }
procedure TFmMain.DefOtherVar;
var
 i, j, m, fromline, tnpar, tnarray:integer;
 tempstring, tname, tunits, tsymbol: string;
// tname, tunits, tsymbol: Tshortstring;
 tptype:processtype;
begin
 fromline := LineSearch(ReModelDef, 'Other Variables');
 // Advance 1 line in the memo to get to the line above the other listing
 fromline := fromline + 2;
 tempstring := ReModelDef.lines[fromline];
 repeat
   tempstring := ReModelDef.lines[fromline];
   tempstring := trim(tempstring);
   fromline := fromline + 1;
 until tempstring <> '';
 i := 1;
 while tempstring <> 'End' do
  begin
   ParseString(tempstring, tname, tunits, tsymbol, tnpar, tptype, tnarray);
   if tname[1] <> '*' then     // Not an array variable
     begin
       m := length(tname);
       delete(tname, stringlength-1, m-stringlength+2);
       tempother[i].name := tname;
       tempother[i].units := tunits;
       tempother[i].symbol := tsymbol;
       i := i + 1;
     end
   else                   // An array variable
     begin
       for j := 1 to tnarray do
         begin
          m := length(tname);
          delete(tname, stringlength-1, m-stringlength+2);
          tempother[i + j - 1].name := tname + inttostr(j);
          tempother[i + j - 1].units := tunits;
          tempother[i + j - 1].symbol := tsymbol {+ '[' + inttostr(j) + ']'};
         end;
       arrays[TotalNumArrays].VarName := tname; // Used in createprocessesproc procedure
       arrays[TotalNumArrays].NumElements := tnarray;
       TotalNumArrays := TotalNumArrays + 1;
       if TotalNumArrays > MaxArrays then
           raise Exception.Create('Too many arrays. Increase MaxArrays.');
       i := i + tnarray;
     end;
   repeat
    tempstring := ReModelDef.lines[fromline];
    tempstring := trim(tempstring);
    fromline := fromline + 1;
   until tempstring <> '';
  end;
 numother := i - 1;
end;

{ This procedure creates the processes procedure in equations.pas. First it
  defines the local variable names using the symbol stored in the arrays. Then
  the value in the global arrays is copied into the local variables. And then
  the model equations are copied from the ReModelDef memo. Finally the values
  of the processes are copied from the local variables to the global variables.
    }
procedure TFmMain.CreateProcessesProc;
var
 toline, fromline1, fromline2, i, j, numprevpar, numemptylines, arraynum,
   linecount, lastline:integer;
 tempstring:string;
 firstparameter:Boolean;
begin
// Define the local variable names

 // State variables and their derivatives.
  toline := LineSearch(MmEquations,'{States}');
  MmEquations.Lines.Delete(toline + 1);   // Delete the example.
  i := 1;
  linecount := 1;
  lastline := toline + linecount;
  repeat
    if tempstate[i].name[1] <> '*' then     // Not an array variable
      begin
        MmEquations.Lines.Insert(toline + linecount, tempstate[i].symbol + ', '
                           + tempprocess[i].symbol + ', ');
        i := i + 1;
        lastline := toline + linecount;
      end
    else    // An array variable
      begin    // Remove comma and set var type to double before adding array.
        if (i <> 1) and (tempstate[i-1].name[1] <> '*') then
          begin
            RemoveTrailingComma(lastline);
            MmEquations.Lines[lastline] := MmEquations.Lines[lastline] + ' :double;';
          end;
        // Add declaration for array variable
        arraynum := GetArrayNum(tempstate[i].name);
        MmEquations.Lines.Insert(toline + linecount, RemoveArrayIndex(tempstate[i].symbol) + ', '
            + RemoveArrayIndex(tempprocess[i].symbol) + ': array[1..' +
            inttostr(arrays[arraynum].NumElements) + '] of double;');
        i := i + arrays[arraynum].NumElements;
        lastline := toline + linecount;
      end;
    linecount := linecount + 1;
  until i = tempModelDef.numstate + 1;

 // Processes and parameters.
  toline := LineSearch(MmEquations,'{processes and associated parameters}');
  MmEquations.Lines.Delete(toline + 1);  // Delete the example.
  linecount := 1;
  lastline := toline;
  i := 1;
  repeat
    if tempprocess[tempmodeldef.numstate + i].name[1] <> '*' then   // Not an array variable
      begin
        tempstring := tempprocess[tempmodeldef.numstate + i].symbol + ', ';
        numprevpar := ParCount(tempmodeldef.numstate + i);
        for j := 1 to tempprocess[tempmodeldef.numstate + i].parameters do
         begin
           tempstring := tempstring + temppar[numprevpar + j].symbol + ', ';
         end;
        MmEquations.Lines.Insert(toline + linecount, tempstring);
        i := i + 1;
        lastline := toline + linecount;
      end
    else         // An array variable
      begin
  // Remove comma and set var type to double before adding array if previous variable was not an array.
        if (i = 1) and (tempstate[tempModelDef.numstate].name[1] <> '*') then
         begin
          RemoveTrailingComma(lastline);
          MmEquations.Lines[lastline] := MmEquations.Lines[lastline] + ' :double;';
         end
        else if (tempprocess[tempmodeldef.numstate + i - 1].name[1] <> '*') then
         begin
          RemoveTrailingComma(lastline);
          MmEquations.Lines[lastline] := MmEquations.Lines[lastline] + ' :double;';
         end;
        // Add declaration for array variable
        arraynum := GetArrayNum(tempprocess[tempmodeldef.numstate + i].name);
        tempstring := tempprocess[tempmodeldef.numstate + i].symbol;
        tempstring := RemoveArrayIndex(tempstring) + ', ';
        numprevpar := ParCount(tempmodeldef.numstate + i);
        for j := 1 to tempprocess[tempmodeldef.numstate + i].parameters do
         begin
           tempstring := tempstring + temppar[numprevpar + j].symbol;
           tempstring := RemoveArrayIndex(tempstring) + ', ';
         end;
        tempstring := trim(tempstring);
        j := length(tempstring);
        if tempstring[j] = ',' then delete(tempstring,j,1);
        tempstring := tempstring + ': array[1..' +
            inttostr(arrays[arraynum].NumElements) + '] of double;';
        MmEquations.Lines.Insert(toline + linecount, tempstring);
        i := i + arrays[arraynum].NumElements;
        lastline := toline + linecount;
      end;
    linecount := linecount + 1;
  until i = tempModelDef.numprocess + 1;

   // Remove the comma after the last variable if there are no other variables.
  if (tempModelDef.numdrive = 0) and (numOther = 0) then
      RemoveTrailingComma(lastline);

 // Driver variables.
  toline := LineSearch(MmEquations,'{drivers}');
  MmEquations.Lines.Delete(toline + 1);   // Delete the example.
  linecount := 1;
  i := 1;
  while i < tempModelDef.numdrive + 1 do
   begin
    if tempdrive[i].name[1] <> '*' then   // Not an array variable
      begin
        MmEquations.Lines.Insert(toline + linecount, tempdrive[i].symbol + ', ');
        i := i + 1;
        lastline := toline + linecount;
      end
    else       // An array variable
      begin
   // Remove comma and set var type to double before adding array if previous variable was not an array.
        if (i = 1) and (tempProcess[tempModelDef.numprocess].name[1] <> '*') then
         begin
          RemoveTrailingComma(lastline);
          MmEquations.Lines[lastline] := MmEquations.Lines[lastline] + ' :double;';
         end
        else if tempdrive[i-1].name[1] <> '*' then
         begin
          RemoveTrailingComma(lastline);
          MmEquations.Lines[lastline] := MmEquations.Lines[lastline] + ' :double;';
         end;
        // Add declaration for array variable
        arraynum := GetArrayNum(tempdrive[i].name);
        MmEquations.Lines.Insert(toline + linecount,
           RemoveArrayIndex(tempdrive[i].symbol) + ': array[1..' +
           inttostr(arrays[arraynum].NumElements) + '] of double;');
        i := i + arrays[arraynum].NumElements;
        lastline := toline + linecount;
      end;
    linecount := linecount + 1;
   end;

// Other variables
  toline := LineSearch(MmEquations,'{other}');
  MmEquations.Lines.Delete(toline + 1);   // Delete the example.
  linecount := 1;
  i := 1;
  if numOther = 0 then      // Clean up because there are no other variables
  begin
   if tempdrive[tempModelDef.numdrive].name[1] <> '*' then
      RemoveTrailingComma(lastline)
   else
      RemoveFinalDouble;
  end
  else     // There are other variables
  begin
   repeat
    if tempother[i].name[1] <> '*' then   // Not an array variable
      begin
        MmEquations.Lines.Insert(toline + linecount, tempother[i].symbol + ', ');
        i := i + 1;
        lastline := toline + linecount;
        if i = numOther + 1 then RemoveTrailingComma(lastline);
      end
    else       // An array variable
      begin    // Remove comma and set var type to double before adding array.
        if tempModelDef.numdrive > 0 then
         begin
          if (i = 1) and (tempDrive[tempModelDef.numdrive].name[1] <> '*') then
           begin
            RemoveTrailingComma(lastline);
            MmEquations.Lines[lastline] := MmEquations.Lines[lastline] + ' :double;';
           end;
         end
        else
         begin
          if (i = 1) and (tempProcess[tempModelDef.numstate + tempModelDef.numprocess].name[1] <> '*') then
           begin
            RemoveTrailingComma(lastline);
            MmEquations.Lines[lastline] := MmEquations.Lines[lastline] + ' :double;';
           end;
         end;
        // Add declaration for array variable
        arraynum := GetArrayNum(tempother[i].name);
        MmEquations.Lines.Insert(toline + linecount,
           tempother[i].symbol + ': array[1..' +
           inttostr(arrays[arraynum].NumElements) + '] of double;');
        i := i + arrays[arraynum].NumElements;
        lastline := toline + linecount;
        if (i = numOther + 1) then RemoveFinalDouble;
      end;
    linecount := linecount + 1;
   until i = numOther + 1;
  end;

// Copy any user defined functions into the processes procedure
  GetFunctions;

// Copy values from the global arrays to the local variables.
 // Drivers
  toline := LineSearch(MmEquations,'{ Copy the drivers from the global array,');
  MmEquations.Lines.Delete(toline + 1);   // Delete the example.
  linecount := 1;
  i := 1;
  while i < tempModelDef.numdrive + 1 do
   begin
    if tempdrive[i].name[1] <> '*' then  // Not an array
      begin
        MmEquations.Lines.Insert(toline + linecount, tempdrive[i].symbol +
           ' := ' + 'tdrive[' + inttostr(i) + '].value;');
        i := i + 1;
      end
    else        // An array variable
      begin
        arraynum := GetArrayNum(tempdrive[i].name);
        tempstring := 'for jj := 1 to ' + inttostr(arrays[arraynum].numElements)
          + ' do ' + RemoveArrayIndex(tempdrive[i].symbol) + '[jj] := tdrive[' + inttostr(i) +
          ' + jj - 1].value;';
        MmEquations.Lines.Insert(toline + linecount, tempstring);
        i := i + arrays[arraynum].numElements;
      end;
    linecount := linecount + 1;
   end;

 // State Variables
  toline := LineSearch(MmEquations,
                           '{ Copy the state variables from the global array');
  MmEquations.Lines.Delete(toline + 1);   // Delete the example.
  linecount := 1;
  i := 1;
  repeat
    if tempstate[i].name[1] <> '*' then  // Not an array
      begin
        MmEquations.Lines.Insert(toline + linecount, tempstate[i].symbol +
           ' := ' + 'tstat[' + inttostr(i) + '].value;');
        i := i + 1;
      end
    else        // An array variable
      begin
        arraynum := GetArrayNum(tempstate[i].name);
        tempstring := 'for jj := 1 to ' + inttostr(arrays[arraynum].numElements)
          + ' do ' + RemoveArrayIndex(tempstate[i].symbol) + '[jj] := tstat[' + inttostr(i-1) +
          ' + jj].value;';
        MmEquations.Lines.Insert(toline + linecount, tempstring);
        i := i + arrays[arraynum].numElements;
      end;
    linecount := linecount + 1;
  until i = tempModelDef.numstate + 1;

 // Parameters
  toline := LineSearch(MmEquations, '  copy the value of the first parameter');
  toline := toline + 2;
  for i := 1 to 2 do MmEquations.Lines.Delete(toline);  // Delete the example.
  i := tempmodeldef.numstate + 1;
  repeat
    firstParameter := True;
    if tempprocess[i].parameters > 0 then
     begin
     if tempprocess[i].name[1] <> '*' then  // Not an array variable
      begin
        numprevpar := ParCount(i);
        for j := 1 to tempprocess[i].parameters do
          begin
            if firstParameter then
              begin  // First parameter of this process
                MmEquations.Lines.Insert(toline,
                   'npar:=ParCount(ModelDef.numstate + ' +
                   inttostr(i-tempmodeldef.numstate) + ');');
                MmEquations.Lines.Insert(toline + 1,
                   temppar[numprevpar + j].symbol + ' := par[npar + ' +
                   inttostr(j) + '].value;');
                firstParameter := False;
                toline := toline + 2;
              end
            else  // Not the first parameter
              begin
                MmEquations.Lines.Insert(toline,
                   temppar[numprevpar + j].symbol + ' := par[npar + ' +
                   inttostr(j) + '].value;');
                toline := toline + 1;
              end;
          end;
        MmEquations.Lines.Insert(toline + 1, ' ');
        toline := toline + 1;
        i := i + 1;
      end
     else        // An array variable
      begin
        numprevpar := ParCount(i);
        for j := 1 to tempprocess[i].parameters do
          begin
            if firstParameter then
              begin  // First parameter of this process
                MmEquations.Lines.Insert(toline,
                  'npar:=ParCount(ModelDef.numstate + ' +
                  inttostr(i-tempmodeldef.numstate) + ');');
                arraynum := GetArrayNum(tempprocess[i].name);
                tempstring := 'for jj := 1 to ' +
                   inttostr(arrays[arraynum].numElements) + ' do ' +
                   RemoveArrayIndex(temppar[numprevpar + j].symbol) +
                   '[jj] := par[npar + ' +
                   inttostr(j) + ' + (jj - 1)*' + inttostr(tempprocess[i].parameters)
                   + '].value;';
                MmEquations.Lines.Insert(toline + 1, tempstring);
                firstParameter := False;
                toline := toline + 2;
              end
            else  // Not the first parameter
              begin
                arraynum := GetArrayNum(tempprocess[i].name);
                tempstring := 'for jj := 1 to ' +
                   inttostr(arrays[arraynum].numElements) + ' do ' +
                   RemoveArrayIndex(temppar[numprevpar + j].symbol) +
                   '[jj] := par[npar + ' +
                   inttostr(j) + ' + (jj - 1)*' + inttostr(tempprocess[i].parameters)
                   + '].value;';
                MmEquations.Lines.Insert(toline, tempstring);
                toline := toline + 1;
              end;
          end;
        MmEquations.Lines.Insert(toline + 1, ' ');
        toline := toline + 1;
        arraynum := GetArrayNum(tempprocess[i].name);
        i := i + arrays[arraynum].numElements;
      end;
     end
    else
     i := i + 1;
  until i = tempmodeldef.numstate +tempmodeldef.numprocess + 1;  

 // Enter the equations.
  toline := LineSearch(MmEquations,
                     '{ Enter the equations to calculate the processes here,');
  toline := toline + 2;
  MmEquations.Lines.Delete(toline); // Delete the example equation.
  fromline1 := LineSearch(ReModelDef, 'Equations');
  fromline2 := LineSearch(ReModelDef, 'End Equations');

  numemptylines := 0;
  i := 1;
  repeat
    tempstring := ReModelDef.Lines[fromline1 + numemptylines + i];
    if tempstring <> 'End Equations' then
        MmEquations.Lines.Insert(toline + i,
              ReModelDef.Lines[fromline1 + numemptylines + i]);
    i := i + 1;
  until i + numemptylines >= fromline2 - fromline1;

// Equations to calculate the derivatives
  toline := LineSearch(MmEquations, 'if tstat[1].HoldConstant');
  fromline1 := LineSearch(ReModelDef, 'Derivatives');

  CheckDerivatives(fromline1);
  // Delete the example code
  for i := 0 to 3 do MmEquations.Lines.Delete(toline);

  // Write the derivatives
  numemptylines := 0;
  linecount := 1;
  i := 1;
  repeat
    repeat
      tempstring := ReModelDef.Lines[fromline1 + numemptylines + linecount];
      tempstring := trim(tempstring);
      linecount := linecount + 1;
    until tempstring <> '';
    if tempstate[i].name[1] <> '*' then  // Not an array variable
      begin
        MmEquations.Lines.Insert(toline, 'if (tstat[' + inttostr(i) +
           '].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then');
        MmEquations.Lines.Insert(toline + 1, ' ' + tempprocess[i].symbol +
          ' := 0');
        MmEquations.Lines.Insert(toline + 2, 'else');
        MmEquations.Lines.Insert(toline + 3, ' ' +
           ReModelDef.Lines[fromline1 + numemptylines + linecount - 1]);
        MmEquations.Lines.Insert(toline + 4, ' ');
        toline := toline + 5;
        i := i + 1;
      end
    else        // An array variable
      begin
        arraynum := GetArrayNum(tempstate[i].name);
        MmEquations.Lines.Insert(toline, 'for jj := 1 to ' +
           inttostr(arrays[arraynum].NumElements) + ' do');
        MmEquations.Lines.Insert(toline + 1, 'begin');
        MmEquations.Lines.Insert(toline + 2, ' if (tstat[' + inttostr(i - 1) +
           ' + jj].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then');
        MmEquations.Lines.Insert(toline + 3, '  ' + RemoveArrayIndex(tempprocess[i].symbol) +
           '[jj] := 0');
        MmEquations.Lines.Insert(toline + 4, ' else');
        MmEquations.Lines.Insert(toline + 5, '  ' +
           ReModelDef.Lines[fromline1 + numemptylines + linecount - 1]);
        MmEquations.Lines.Insert(toline + 6, 'end;');
        MmEquations.Lines.Insert(toline + 7, ' ');
        toline := toline + 8;
        i := i + arrays[arraynum].NumElements;
      end;
  until i = tempmodeldef.numstate + 1;

// Copy the new values from the local variables back into the global arrays.
 // State variables
  toline := LineSearch(MmEquations, 'tstat[1].value := ');
  MmEquations.Lines.Delete(toline);  // Delete the example.

  i := 1;
  linecount := 1;
  repeat
   if tempstate[i].name[1] <> '*' then    // Not an array variable
    begin
      MmEquations.Lines.Insert(toline + linecount - 1, 'tstat[' + inttostr(i) +
         '].value := ' + tempstate[i].symbol + ';');
      i := i + 1;
    end
   else             // An array variable
    begin
      arraynum := GetArrayNum(tempstate[i].name);
      MmEquations.Lines.Insert(toline + linecount - 1, 'for jj := 1 to ' +
        inttostr(arrays[arraynum].NumElements) + ' do tstat[' + inttostr(i-1) +
        ' + jj].value := ' + RemoveArrayIndex(tempstate[i].symbol) + '[jj];');
      i := i + arrays[arraynum].NumElements;
    end;
   linecount := linecount + 1;
  until i = tempmodeldef.numstate + 1;

 // State variable derivatives
  toline := LineSearch(MmEquations, 'tproc[1].value := ');
  MmEquations.Lines.Delete(toline);  // Delete the example.

  i := 1;
  linecount := 1;
  repeat
   if tempstate[i].name[1] <> '*' then    // Not an array variable
    begin
      MmEquations.Lines.Insert(toline + linecount - 1, 'tproc[' + inttostr(i) +
         '].value := ' + tempprocess[i].symbol + ';');
      i := i + 1;
    end
   else             // An array variable
    begin
      arraynum := GetArrayNum(tempstate[i].name);
      MmEquations.Lines.Insert(toline + linecount - 1, 'for jj := 1 to ' +
        inttostr(arrays[arraynum].NumElements) + ' do tproc[' + inttostr(i-1) +
        ' + jj].value := ' + RemoveArrayIndex(tempprocess[i].symbol) + '[jj];');
      i := i + arrays[arraynum].NumElements;
    end;
   linecount := linecount + 1;
  until i = tempmodeldef.numstate + 1;

 // Processes
  toline := LineSearch(MmEquations, 'tproc[ModelDef.numstate + 1].value :=');
  MmEquations.Lines.Delete(toline);  // Delete the example.

  i := 1;
  linecount := 1;
  repeat
   if tempprocess[tempmodeldef.numstate + i].name[1] <> '*' then    // Not an array variable
    begin
      MmEquations.Lines.Insert(toline + linecount - 1, 'tproc[ModelDef.numstate + ' +
                inttostr(i) + '].value := ' +
                tempprocess[tempmodeldef.numstate + i].symbol + ';');
      i := i + 1;
    end
   else       // An array variable
    begin
      arraynum := GetArrayNum(tempprocess[tempmodeldef.numstate + i].name);
      MmEquations.Lines.Insert(toline + linecount - 1, 'for jj := 1 to ' +
         inttostr(arrays[arraynum].NumElements) + ' do tproc[ModelDef.numstate + ' +
         inttostr(i - 1) + ' + jj].value := ' +
         RemoveArrayIndex(tempprocess[tempmodeldef.numstate + i].symbol) + '[jj];');
      i := i + arrays[arraynum].NumElements;
    end;
   linecount := linecount + 1;
  until i = tempmodeldef.numprocess + 1;
end;


{ Searches tempmemo for the string Sstring and returns the line number
  containing Sstring. }
function TFmMain.LineSearch(tempmemo: TCustomMemo; Sstring:string):integer;
var
 i,temp:integer;
begin
 i := 0;
 temp := 0;
 while (i < tempmemo.lines.count - 1) and (temp = 0) do
  begin
    temp := pos(Sstring,tempmemo.lines[i]);
    i := i + 1;
  end;
 linesearch := i - 1;
end;

{ Divides the string, fromString, into two parts at the string, divider. Returns
  either the beginning or the end of the string based on the value of WhichPart.
  The string divider is not included in either part. }
function TFmMain.DivideString(fromString, divider:string;
                             WhichPart:TPart):string;
var
 whereDivide, slength: integer;
begin
 whereDivide := pos(divider, fromString);
 slength := length(fromString);
 if WhichPart = stBeginning then
   begin
     delete(fromString,whereDivide,slength-whereDivide+1);
     fromString := trim(fromString);
   end
 else
   begin
     delete(fromString,1,whereDivide);
     fromString := trim(fromString);
   end;
 DivideString := fromString;
end;

{ Separates tstring into the name, units, symbol, and npar based on
  the position of the commas. }
procedure TFmMain.ParseString(var tstring, tname, tunits, tsymbol: string;
                          var tnpar:integer; var tptype:processtype;
                          var narray: integer);
var
 num, num2:integer;
 tstring2:string;
begin
 num := pos(',',tstring);          // Name
 tname := DivideString(tstring, ',', stBeginning);
 delete(tstring, 1, num);

 num := pos(',',tstring);          // Units
 tunits := DivideString(tstring, ',', stBeginning);
 delete(tstring, 1, num);

 num := pos(',', tstring);         // Symbol
 tsymbol := DivideString(tstring, ',', stBeginning);

 if num = 0 then               // Not an array variable and not a process
  begin
   tnpar := 0;
  end
 else                          // An array variable, a process, or both.
  begin
   delete(tstring, 1, num);

   num := pos(',',tstring);
   narray := strtoint(DivideString(tstring, ',', stBeginning));
   delete(tstring, 1, num);

   if num <> 0 then                     // A process
     begin
       tnpar := narray;
       narray := 0;
       tstring2 := DivideString(tstring, ',', stBeginning);
       num2 := length('Group');
       delete(tstring2, 1, num2);
       num2 := strtoint(tstring2);
       case num2 of
          1: tptype := ptGroup1;
          2: tptype := ptGroup2;
          3: tptype := ptGroup3;
          4: tptype := ptGroup4;
          5: tptype := ptGroup5;
       else
          tptype := ptGroup1;
       end;
       num := pos(',', tstring);
       if num <> 0 then         // A process array
        begin
          delete(tstring, 1, num);
          narray := strtoint(tstring);
        end;
     end;
  end;
end;

{This function counts the parameters in all processes less than processnum.}
function TFmMain.ParCount(processnum:integer):integer;
var
 NumberofParams, counter : integer;
begin
  NumberofParams := 0;
  for counter := tempModelDef.numstate + 1 to processnum - 1 do
    begin
         NumberofParams := NumberofParams + tempprocess[counter].parameters;
    end;
  ParCount := NumberofParams;
end; // end of parcount function

procedure TFmMain.BtnCloseClick(Sender: TObject);
begin
  FmMain.Close;
end;

procedure TFmMain.FormCreate(Sender: TObject);
begin
 FExePath := ExtractFilePath(Application.EXEName);
 DlgOpenModelDef.InitialDir := FExePath;
end;

function TFmMain.GetArrayNum(varname:string): integer;
var
 i, number:integer;
begin
 number := length(varname);
 delete(varname,number,1);          
 number := 0;
 for i := 1 to TotalNumArrays do
    begin
      if varname = arrays[i].VarName then number := i;
    end;
 if number = 0 then
   begin
     delete(varname,length(varname),1);
     for i := 1 to TotalNumArrays do
       if varname = arrays[i].varname then number := i;
   end;

 if number <> 0 then
   GetArrayNum := number
 else
   raise Exception.Create('Could not find variable, ' + varname + ' in arrays array.');
end;

procedure TFmMain.GetFunctions;
var
 fromline, toline, numsourcelines, i: integer;
begin
 fromline := LineSearch(ReModelDef, 'Functions');
 toline := LineSearch(ReModelDef, 'End Functions');
 numsourcelines := toline - fromline;

 toline := LineSearch(MmEquations,'{ Functions or procedures }');
 if numsourcelines > 3 then
   begin
     i := 1;
     repeat
       MmEquations.Lines.Insert(toline + i, ReModelDef.Lines[fromline + i]);
       i := i + 1;
     until i = numsourcelines;
   end;
end;

function TFmMain.CheckDerivatives(fline: integer): integer;
var
 i, j, arraynum: integer;
 tempstring, tempstring2: string;
begin
 i := 1;
 fline := fline + 1;
 repeat
  repeat
   tempstring := ReModelDef.Lines[fline];
   fline := fline + 1;
  until tempstring <> '';
  j := pos(':',tempstring);
  delete(tempstring, j, length(tempstring)-j+1);
  tempstring := trim(tempstring);
  if tempprocess[i].name[1] = '*' then
    begin
     j := pos('[',tempstring);
     delete(tempstring, j, length(tempstring)-j+1);
     tempstring2 := tempprocess[i].symbol;
     delete(tempstring2,pos('[',tempstring2),length(tempstring2));
     if lowercase(tempstring2) <> lowercase(tempstring) then
       raise Exception.Create('Derivatives out of order. Model translation failed.');
     arraynum := GetArrayNum(tempstate[i].name);
     i := i + arrays[arraynum].NumElements;
    end
  else
    begin
     if lowercase(tempprocess[i].symbol) <>
      lowercase(tempstring) then
     raise Exception.Create('Derivatives out of order. Model translation failed.');
     i := i + 1;
    end;
 until i = tempmodeldef.numstate + 1;
 CheckDerivatives := 1;
end;

function TFmMain.RemoveTrailingComma(lnum: integer): integer;
var
 tempstring: string;
 ll: integer;
begin
 RemoveTrailingComma := 0;
 tempstring := MmEquations.Lines[lnum];
 tempstring := trim(tempstring);
 ll := length(tempstring);
 if tempstring[ll] = ',' then delete(tempstring, ll, 1);
 MmEquations.Lines[lnum] := tempstring;
 RemoveTrailingComma := 1;
end;

procedure TFmMain.RemoveFinalDouble;
var
 ll: integer;
begin
 ll := LineSearch(Mmequations,':double; {Final double}');
 MmEquations.Lines[ll] := '';
end;

function TFmMain.RemoveArrayIndex(tstring: string): string;
var
 i: integer;
begin
 i := pos('[', tstring);
 delete(tstring, i, length(tstring) - 1);
 RemoveArrayIndex := tstring;
end;

end.
